/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.azure;

import com.azure.core.http.rest.ResponseBase;
import com.azure.core.util.BinaryData;
import com.azure.storage.blob.BlobAsyncClient;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerAsyncClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceAsyncClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.BlobItemProperties;
import com.azure.storage.blob.models.BlobListDetails;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.models.BlobRequestConditions;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.models.DownloadRetryOptions;
import com.azure.storage.blob.models.ListBlobsOptions;
import com.azure.storage.blob.options.BlobParallelUploadOptions;
import com.azure.storage.blob.options.BlockBlobSimpleUploadOptions;
import com.azure.storage.blob.specialized.BlobClientBase;
import com.azure.storage.blob.specialized.BlobLeaseClient;
import com.azure.storage.blob.specialized.BlobLeaseClientBuilder;
import com.azure.storage.blob.specialized.BlockBlobAsyncClient;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.util.ReferenceCountUtil;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.file.FileAlreadyExistsException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Spliterators;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.BiPredicate;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.util.Throwables;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.OptionalBytesReference;
import org.elasticsearch.common.blobstore.support.BlobContainerUtils;
import org.elasticsearch.common.blobstore.support.BlobMetadata;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.repositories.azure.AzureBlobContainer;
import org.elasticsearch.repositories.azure.AzureBlobServiceClient;
import org.elasticsearch.repositories.azure.AzureRepository;
import org.elasticsearch.repositories.azure.AzureStorageService;
import org.elasticsearch.repositories.azure.CancellableRateLimitedFluxIterator;
import org.elasticsearch.repositories.azure.LocationMode;
import org.elasticsearch.repositories.azure.SocketAccess;
import org.elasticsearch.repositories.blobstore.ChunkedBlobOutputStream;
import org.elasticsearch.rest.RestStatus;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

public class AzureBlobStore
implements BlobStore {
    private static final Logger logger = LogManager.getLogger(AzureBlobStore.class);
    private static final long DEFAULT_READ_CHUNK_SIZE = new ByteSizeValue(32L, ByteSizeUnit.MB).getBytes();
    private static final int DEFAULT_UPLOAD_BUFFERS_SIZE = (int)new ByteSizeValue(64L, ByteSizeUnit.KB).getBytes();
    private final AzureStorageService service;
    private final BigArrays bigArrays;
    private final String clientName;
    private final String container;
    private final LocationMode locationMode;
    private final ByteSizeValue maxSinglePartUploadSize;
    private final Stats stats = new Stats();
    private final BiConsumer<String, URL> statsConsumer;
    private static final int CONCURRENT_DELETES = 100;
    private static final Base64.Encoder base64Encoder = Base64.getEncoder().withoutPadding();
    private static final Base64.Decoder base64UrlDecoder = Base64.getUrlDecoder();

    public AzureBlobStore(RepositoryMetadata metadata, AzureStorageService service, BigArrays bigArrays) {
        this.container = (String)AzureRepository.Repository.CONTAINER_SETTING.get(metadata.settings());
        this.clientName = (String)AzureRepository.Repository.CLIENT_NAME.get(metadata.settings());
        this.service = service;
        this.bigArrays = bigArrays;
        this.locationMode = (LocationMode)((Object)AzureRepository.Repository.LOCATION_MODE_SETTING.get(metadata.settings()));
        this.maxSinglePartUploadSize = (ByteSizeValue)AzureRepository.Repository.MAX_SINGLE_PART_UPLOAD_SIZE_SETTING.get(metadata.settings());
        List<RequestStatsCollector> requestStatsCollectors = List.of(RequestStatsCollector.create((httpMethod, url) -> httpMethod.equals("HEAD"), this.stats.headOperations::incrementAndGet), RequestStatsCollector.create((httpMethod, url) -> httpMethod.equals("GET") && !this.isListRequest((String)httpMethod, (URL)url), this.stats.getOperations::incrementAndGet), RequestStatsCollector.create(this::isListRequest, this.stats.listOperations::incrementAndGet), RequestStatsCollector.create(this::isPutBlockRequest, this.stats.putBlockOperations::incrementAndGet), RequestStatsCollector.create(this::isPutBlockListRequest, this.stats.putBlockListOperations::incrementAndGet), RequestStatsCollector.create((httpMethod, url) -> httpMethod.equals("PUT") && !this.isPutBlockRequest((String)httpMethod, (URL)url) && !this.isPutBlockListRequest((String)httpMethod, (URL)url), this.stats.putOperations::incrementAndGet));
        this.statsConsumer = (httpMethod, url) -> {
            try {
                String path;
                URI uri = url.toURI();
                String string = path = uri.getPath() == null ? "" : uri.getPath();
                if (!path.contains(this.container)) {
                    return;
                }
                assert (path.contains(this.container)) : uri.toString();
            }
            catch (URISyntaxException ignored) {
                return;
            }
            for (RequestStatsCollector requestStatsCollector : requestStatsCollectors) {
                if (!requestStatsCollector.shouldConsumeRequestInfo((String)httpMethod, (URL)url)) continue;
                requestStatsCollector.consumeHttpRequestInfo();
                return;
            }
        };
    }

    private boolean isListRequest(String httpMethod, URL url) {
        return httpMethod.equals("GET") && url.getQuery() != null && url.getQuery().contains("comp=list");
    }

    private boolean isPutBlockRequest(String httpMethod, URL url) {
        String queryParams = url.getQuery() == null ? "" : url.getQuery();
        return httpMethod.equals("PUT") && queryParams.contains("comp=block") && queryParams.contains("blockid=");
    }

    private boolean isPutBlockListRequest(String httpMethod, URL url) {
        String queryParams = url.getQuery() == null ? "" : url.getQuery();
        return httpMethod.equals("PUT") && queryParams.contains("comp=blocklist");
    }

    public long getReadChunkSize() {
        return DEFAULT_READ_CHUNK_SIZE;
    }

    public String toString() {
        return this.container;
    }

    public AzureStorageService getService() {
        return this.service;
    }

    public LocationMode getLocationMode() {
        return this.locationMode;
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new AzureBlobContainer(path, this);
    }

    public void close() {
    }

    public boolean blobExists(String blob) throws IOException {
        BlobServiceClient client = this.client();
        try {
            Boolean blobExists = SocketAccess.doPrivilegedException(() -> {
                BlobClient azureBlob = client.getBlobContainerClient(this.container).getBlobClient(blob);
                return azureBlob.exists();
            });
            return Boolean.TRUE.equals(blobExists);
        }
        catch (Exception e) {
            logger.trace("can not access [{}] in container {{}}: {}", (Object)blob, (Object)this.container, (Object)e.getMessage());
            throw new IOException("Unable to check if blob " + blob + " exists", e);
        }
    }

    public DeleteResult deleteBlobDirectory(String path) throws IOException {
        AtomicInteger blobsDeleted = new AtomicInteger(0);
        AtomicLong bytesDeleted = new AtomicLong(0L);
        SocketAccess.doPrivilegedVoidException(() -> {
            BlobContainerAsyncClient blobContainerAsyncClient = this.asyncClient().getBlobContainerAsyncClient(this.container);
            ListBlobsOptions options = new ListBlobsOptions().setPrefix(path).setDetails(new BlobListDetails().setRetrieveMetadata(true));
            try {
                blobContainerAsyncClient.listBlobs(options, null).flatMap(blobItem -> {
                    if (blobItem.isPrefix() != null && blobItem.isPrefix().booleanValue()) {
                        return Mono.empty();
                    }
                    String blobName = blobItem.getName();
                    BlobAsyncClient blobAsyncClient = blobContainerAsyncClient.getBlobAsyncClient(blobName);
                    Mono<Void> deleteTask = AzureBlobStore.getDeleteTask(blobName, blobAsyncClient);
                    bytesDeleted.addAndGet(blobItem.getProperties().getContentLength());
                    blobsDeleted.incrementAndGet();
                    return deleteTask;
                }, 100).then().block();
            }
            catch (Exception e) {
                AzureBlobStore.filterDeleteExceptionsAndRethrow(e, new IOException("Deleting directory [" + path + "] failed"));
            }
        });
        return new DeleteResult((long)blobsDeleted.get(), bytesDeleted.get());
    }

    private static void filterDeleteExceptionsAndRethrow(Exception e, IOException exception) throws IOException {
        int suppressedCount = 0;
        for (Throwable suppressed : e.getSuppressed()) {
            if (!(suppressed instanceof IOException)) continue;
            exception.addSuppressed(suppressed);
            if (++suppressedCount > 10) break;
        }
        throw exception;
    }

    void deleteBlobs(Iterator<String> blobs) throws IOException {
        if (!blobs.hasNext()) {
            return;
        }
        BlobServiceAsyncClient asyncClient = this.asyncClient();
        SocketAccess.doPrivilegedVoidException(() -> {
            BlobContainerAsyncClient blobContainerClient = asyncClient.getBlobContainerAsyncClient(this.container);
            try {
                Flux.fromStream(StreamSupport.stream(Spliterators.spliteratorUnknownSize(blobs, 16), false)).flatMap(blob -> AzureBlobStore.getDeleteTask(blob, blobContainerClient.getBlobAsyncClient(blob)), 100).then().block();
            }
            catch (Exception e) {
                AzureBlobStore.filterDeleteExceptionsAndRethrow(e, new IOException("Unable to delete blobs"));
            }
        });
    }

    private static Mono<Void> getDeleteTask(String blobName, BlobAsyncClient blobAsyncClient) {
        return blobAsyncClient.delete().onErrorResume(e -> {
            BlobStorageException blobStorageException;
            return e instanceof BlobStorageException && (blobStorageException = (BlobStorageException)e).getStatusCode() == 404;
        }, throwable -> Mono.empty()).onErrorMap(throwable -> new IOException("Error deleting blob " + blobName, (Throwable)throwable));
    }

    public InputStream getInputStream(String blob, long position, @Nullable Long length) throws IOException {
        logger.trace(() -> Strings.format((String)"reading container [%s], blob [%s]", (Object[])new Object[]{this.container, blob}));
        AzureBlobServiceClient azureBlobServiceClient = this.getAzureBlobServiceClientClient();
        BlobServiceClient syncClient = azureBlobServiceClient.getSyncClient();
        BlobServiceAsyncClient asyncClient = azureBlobServiceClient.getAsyncClient();
        return SocketAccess.doPrivilegedException(() -> {
            BlobContainerClient blobContainerClient = syncClient.getBlobContainerClient(this.container);
            BlobClient blobClient = blobContainerClient.getBlobClient(blob);
            long totalSize = length == null ? blobClient.getProperties().getBlobSize() : position + length;
            BlobAsyncClient blobAsyncClient = asyncClient.getBlobContainerAsyncClient(this.container).getBlobAsyncClient(blob);
            int maxReadRetries = this.service.getMaxReadRetries(this.clientName);
            return new AzureInputStream(blobAsyncClient, position, length == null ? totalSize : length, totalSize, maxReadRetries, azureBlobServiceClient.getAllocator());
        });
    }

    public Map<String, BlobMetadata> listBlobsByPrefix(String keyPath, String prefix) throws IOException {
        HashMap blobsBuilder = new HashMap();
        logger.trace(() -> Strings.format((String)"listing container [%s], keyPath [%s], prefix [%s]", (Object[])new Object[]{this.container, keyPath, prefix}));
        try {
            BlobServiceClient client = this.client();
            SocketAccess.doPrivilegedVoidException(() -> {
                BlobContainerClient containerClient = client.getBlobContainerClient(this.container);
                BlobListDetails details = new BlobListDetails().setRetrieveMetadata(true);
                ListBlobsOptions listBlobsOptions = new ListBlobsOptions().setPrefix(keyPath + (prefix == null ? "" : prefix)).setDetails(details);
                for (BlobItem blobItem : containerClient.listBlobsByHierarchy("/", listBlobsOptions, null)) {
                    BlobItemProperties properties = blobItem.getProperties();
                    Boolean isPrefix = blobItem.isPrefix();
                    if (isPrefix != null && isPrefix.booleanValue()) continue;
                    String blobName = blobItem.getName().substring(keyPath.length());
                    blobsBuilder.put(blobName, new BlobMetadata(blobName, properties.getContentLength().longValue()));
                }
            });
        }
        catch (Exception e) {
            throw new IOException("Unable to list blobs by prefix [" + prefix + "] for path " + keyPath, e);
        }
        return Map.copyOf(blobsBuilder);
    }

    public Map<String, BlobContainer> children(BlobPath path) throws IOException {
        HashMap childrenBuilder = new HashMap();
        String keyPath = path.buildAsString();
        try {
            BlobServiceClient client = this.client();
            SocketAccess.doPrivilegedVoidException(() -> {
                BlobContainerClient blobContainer = client.getBlobContainerClient(this.container);
                ListBlobsOptions listBlobsOptions = new ListBlobsOptions();
                listBlobsOptions.setPrefix(keyPath).setDetails(new BlobListDetails().setRetrieveMetadata(true));
                for (BlobItem blobItem : blobContainer.listBlobsByHierarchy("/", listBlobsOptions, null)) {
                    Boolean isPrefix = blobItem.isPrefix();
                    if (isPrefix == null || !isPrefix.booleanValue()) continue;
                    String directoryName = blobItem.getName();
                    if ((directoryName = directoryName.substring(keyPath.length())).isEmpty()) continue;
                    directoryName = directoryName.substring(0, directoryName.length() - 1);
                    childrenBuilder.put(directoryName, new AzureBlobContainer(BlobPath.EMPTY.add(blobItem.getName()), this));
                }
            });
        }
        catch (Exception e) {
            throw new IOException("Unable to provide children blob containers for " + path, e);
        }
        return Collections.unmodifiableMap(childrenBuilder);
    }

    public void writeBlob(String blobName, BytesReference bytes, boolean failIfAlreadyExists) {
        Flux byteBufferFlux = Flux.fromArray((Object[])BytesReference.toByteBuffers((BytesReference)bytes));
        this.executeSingleUpload(blobName, (Flux<ByteBuffer>)byteBufferFlux, bytes.length(), failIfAlreadyExists);
    }

    public void writeBlob(final String blobName, final boolean failIfAlreadyExists, CheckedConsumer<OutputStream, IOException> writer) throws IOException {
        final BlockBlobAsyncClient blockBlobAsyncClient = this.asyncClient().getBlobContainerAsyncClient(this.container).getBlobAsyncClient(blobName).getBlockBlobAsyncClient();
        try (ChunkedBlobOutputStream<String> out = new ChunkedBlobOutputStream<String>(this.bigArrays, this.getUploadBlockSize()){

            protected void flushBuffer() {
                if (this.buffer.size() == 0) {
                    return;
                }
                String blockId = AzureBlobStore.this.makeMultipartBlockId();
                SocketAccess.doPrivilegedVoidException(() -> blockBlobAsyncClient.stageBlock(blockId, Flux.fromArray((Object[])BytesReference.toByteBuffers((BytesReference)this.buffer.bytes())), (long)this.buffer.size()).block());
                this.finishPart(blockId);
            }

            protected void onCompletion() {
                if (this.flushedBytes == 0L) {
                    AzureBlobStore.this.writeBlob(blobName, this.buffer.bytes(), failIfAlreadyExists);
                } else {
                    this.flushBuffer();
                    SocketAccess.doPrivilegedVoidException(() -> blockBlobAsyncClient.commitBlockList(this.parts, !failIfAlreadyExists).block());
                }
            }

            protected void onFailure() {
            }
        };){
            writer.accept((Object)out);
            out.markSuccess();
        }
    }

    public void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        assert (inputStream.markSupported()) : "Should not be used with non-mark supporting streams as their retry handling in the SDK is broken";
        logger.trace(() -> Strings.format((String)"writeBlob(%s, stream, %s)", (Object[])new Object[]{blobName, blobSize}));
        try {
            if (blobSize <= this.getLargeBlobThresholdInBytes()) {
                Flux<ByteBuffer> byteBufferFlux = this.convertStreamToByteBuffer(inputStream, blobSize, DEFAULT_UPLOAD_BUFFERS_SIZE);
                this.executeSingleUpload(blobName, byteBufferFlux, blobSize, failIfAlreadyExists);
            } else {
                this.executeMultipartUpload(blobName, inputStream, blobSize, failIfAlreadyExists);
            }
        }
        catch (BlobStorageException e) {
            if (failIfAlreadyExists && e.getStatusCode() == 409 && BlobErrorCode.BLOB_ALREADY_EXISTS.equals((Object)e.getErrorCode())) {
                throw new FileAlreadyExistsException(blobName, null, e.getMessage());
            }
            throw new IOException("Unable to write blob " + blobName, e);
        }
        catch (Exception e) {
            throw new IOException("Unable to write blob " + blobName, e);
        }
        logger.trace(() -> Strings.format((String)"writeBlob(%s, stream, %s) - done", (Object[])new Object[]{blobName, blobSize}));
    }

    private void executeSingleUpload(String blobName, Flux<ByteBuffer> byteBufferFlux, long blobSize, boolean failIfAlreadyExists) {
        SocketAccess.doPrivilegedVoidException(() -> {
            BlobServiceAsyncClient asyncClient = this.asyncClient();
            BlobAsyncClient blobAsyncClient = asyncClient.getBlobContainerAsyncClient(this.container).getBlobAsyncClient(blobName);
            BlockBlobAsyncClient blockBlobAsyncClient = blobAsyncClient.getBlockBlobAsyncClient();
            BlockBlobSimpleUploadOptions options = new BlockBlobSimpleUploadOptions(byteBufferFlux, blobSize);
            BlobRequestConditions requestConditions = new BlobRequestConditions();
            if (failIfAlreadyExists) {
                requestConditions.setIfNoneMatch("*");
            }
            options.setRequestConditions(requestConditions);
            blockBlobAsyncClient.uploadWithResponse(options).block();
        });
    }

    private void executeMultipartUpload(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) {
        SocketAccess.doPrivilegedVoidException(() -> {
            BlobServiceAsyncClient asyncClient = this.asyncClient();
            BlobAsyncClient blobAsyncClient = asyncClient.getBlobContainerAsyncClient(this.container).getBlobAsyncClient(blobName);
            BlockBlobAsyncClient blockBlobAsyncClient = blobAsyncClient.getBlockBlobAsyncClient();
            long partSize = this.getUploadBlockSize();
            Tuple<Long, Long> multiParts = AzureBlobStore.numberOfMultiparts(blobSize, partSize);
            int nbParts = ((Long)multiParts.v1()).intValue();
            long lastPartSize = (Long)multiParts.v2();
            assert (blobSize == (long)(nbParts - 1) * partSize + lastPartSize) : "blobSize does not match multipart sizes";
            ArrayList<String> blockIds = new ArrayList<String>(nbParts);
            for (int i = 0; i < nbParts; ++i) {
                long length = i < nbParts - 1 ? partSize : lastPartSize;
                Flux<ByteBuffer> byteBufferFlux = this.convertStreamToByteBuffer(inputStream, length, DEFAULT_UPLOAD_BUFFERS_SIZE);
                String blockId = this.makeMultipartBlockId();
                blockBlobAsyncClient.stageBlock(blockId, byteBufferFlux, length).block();
                blockIds.add(blockId);
            }
            blockBlobAsyncClient.commitBlockList(blockIds, !failIfAlreadyExists).block();
        });
    }

    private String makeMultipartBlockId() {
        return base64Encoder.encodeToString(base64UrlDecoder.decode(UUIDs.base64UUID()));
    }

    private Flux<ByteBuffer> convertStreamToByteBuffer(InputStream delegate, long length, int chunkSize) {
        assert (delegate.markSupported()) : "An InputStream with mark support was expected";
        FilterInputStream inputStream = new FilterInputStream(delegate){

            @Override
            public synchronized int read(byte[] b, int off, int len) throws IOException {
                return super.read(b, off, len);
            }

            @Override
            public synchronized int read() throws IOException {
                return super.read();
            }
        };
        ((InputStream)inputStream).mark(Integer.MAX_VALUE);
        return Flux.defer(() -> {
            AtomicLong currentTotalLength = new AtomicLong(0L);
            try {
                inputStream.reset();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            int parts = (int)length / chunkSize;
            long remaining = length % (long)chunkSize;
            return Flux.range((int)0, (int)(remaining == 0L ? parts : parts + 1)).map(i -> i * chunkSize).concatMap(pos -> Mono.fromCallable(() -> {
                long count = (long)(pos + chunkSize) > length ? length - (long)pos.intValue() : (long)chunkSize;
                int numOfBytesRead = 0;
                int offset = 0;
                int len = (int)count;
                byte[] buffer = new byte[len];
                while (numOfBytesRead != -1 && (long)offset < count) {
                    numOfBytesRead = inputStream.read(buffer, offset, len);
                    offset += numOfBytesRead;
                    len -= numOfBytesRead;
                    if (numOfBytesRead == -1) continue;
                    currentTotalLength.addAndGet(numOfBytesRead);
                }
                if (numOfBytesRead == -1 && currentTotalLength.get() < length) {
                    throw new IllegalStateException("InputStream provided" + currentTotalLength + " bytes, less than the expected" + length + " bytes");
                }
                return ByteBuffer.wrap(buffer);
            })).doOnComplete(() -> {
                if (currentTotalLength.get() > length) {
                    throw new IllegalStateException("Read more data than was requested. Size of data read: " + currentTotalLength.get() + ". Size of data requested: " + length);
                }
            });
        }).subscribeOn(Schedulers.elastic());
    }

    static Tuple<Long, Long> numberOfMultiparts(long totalSize, long partSize) {
        if (partSize <= 0L) {
            throw new IllegalArgumentException("Part size must be greater than zero");
        }
        if (totalSize == 0L || totalSize <= partSize) {
            return Tuple.tuple((Object)1L, (Object)totalSize);
        }
        long parts = totalSize / partSize;
        long remaining = totalSize % partSize;
        if (remaining == 0L) {
            return Tuple.tuple((Object)parts, (Object)partSize);
        }
        return Tuple.tuple((Object)(parts + 1L), (Object)remaining);
    }

    long getLargeBlobThresholdInBytes() {
        return this.maxSinglePartUploadSize.getBytes();
    }

    long getUploadBlockSize() {
        return this.service.getUploadBlockSize();
    }

    private BlobServiceClient client() {
        return this.getAzureBlobServiceClientClient().getSyncClient();
    }

    private BlobServiceAsyncClient asyncClient() {
        return this.getAzureBlobServiceClientClient().getAsyncClient();
    }

    private AzureBlobServiceClient getAzureBlobServiceClientClient() {
        return this.service.client(this.clientName, this.locationMode, this.statsConsumer);
    }

    public Map<String, Long> stats() {
        return this.stats.toMap();
    }

    OptionalBytesReference getRegister(String blobPath, String containerPath, String blobKey) {
        try {
            return SocketAccess.doPrivilegedException(() -> OptionalBytesReference.of((BytesReference)AzureBlobStore.downloadRegisterBlob(containerPath, blobKey, this.getAzureBlobServiceClientClient().getSyncClient().getBlobContainerClient(this.container).getBlobClient(blobPath), null)));
        }
        catch (Exception e) {
            BlobStorageException blobStorageException;
            Throwable throwable = Throwables.getRootCause((Throwable)e);
            if (throwable instanceof BlobStorageException && (blobStorageException = (BlobStorageException)throwable).getStatusCode() == RestStatus.NOT_FOUND.getStatus()) {
                return OptionalBytesReference.MISSING;
            }
            throw e;
        }
    }

    OptionalBytesReference compareAndExchangeRegister(String blobPath, String containerPath, String blobKey, BytesReference expected, BytesReference updated) {
        BlobContainerUtils.ensureValidRegisterContent((BytesReference)updated);
        try {
            return SocketAccess.doPrivilegedException(() -> OptionalBytesReference.of((BytesReference)AzureBlobStore.innerCompareAndExchangeRegister(containerPath, blobKey, this.getAzureBlobServiceClientClient().getSyncClient().getBlobContainerClient(this.container).getBlobClient(blobPath), expected, updated)));
        }
        catch (Exception e) {
            BlobStorageException blobStorageException;
            Throwable throwable = Throwables.getRootCause((Throwable)e);
            if (throwable instanceof BlobStorageException && ((blobStorageException = (BlobStorageException)throwable).getStatusCode() == RestStatus.PRECONDITION_FAILED.getStatus() || blobStorageException.getStatusCode() == RestStatus.CONFLICT.getStatus())) {
                return OptionalBytesReference.MISSING;
            }
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static BytesReference innerCompareAndExchangeRegister(String containerPath, String blobKey, BlobClient blobClient, BytesReference expected, BytesReference updated) throws IOException {
        if (blobClient.exists().booleanValue()) {
            BlobLeaseClient leaseClient = new BlobLeaseClientBuilder().blobClient((BlobClientBase)blobClient).buildClient();
            String leaseId = leaseClient.acquireLease(60);
            try {
                BytesReference currentValue = AzureBlobStore.downloadRegisterBlob(containerPath, blobKey, blobClient, new BlobRequestConditions().setLeaseId(leaseId));
                if (currentValue.equals(expected)) {
                    AzureBlobStore.uploadRegisterBlob(updated, blobClient, new BlobRequestConditions().setLeaseId(leaseId));
                }
                BytesReference bytesReference = currentValue;
                return bytesReference;
            }
            finally {
                leaseClient.releaseLease();
            }
        }
        if (expected.length() == 0) {
            AzureBlobStore.uploadRegisterBlob(updated, blobClient, new BlobRequestConditions().setIfNoneMatch("*"));
        }
        return BytesArray.EMPTY;
    }

    private static BytesReference downloadRegisterBlob(String containerPath, String blobKey, BlobClient blobClient, BlobRequestConditions blobRequestConditions) throws IOException {
        return BlobContainerUtils.getRegisterUsingConsistentRead((InputStream)((BinaryData)blobClient.downloadContentWithResponse(new DownloadRetryOptions().setMaxRetryRequests(0), blobRequestConditions, null, null).getValue()).toStream(), (String)containerPath, (String)blobKey);
    }

    private static void uploadRegisterBlob(BytesReference blobContents, BlobClient blobClient, BlobRequestConditions requestConditions) throws IOException {
        blobClient.uploadWithResponse(new BlobParallelUploadOptions(BinaryData.fromStream((InputStream)blobContents.streamInput(), (Long)Long.valueOf(blobContents.length()))).setRequestConditions(requestConditions), null, null);
    }

    private static class Stats {
        private final AtomicLong getOperations = new AtomicLong();
        private final AtomicLong listOperations = new AtomicLong();
        private final AtomicLong headOperations = new AtomicLong();
        private final AtomicLong putOperations = new AtomicLong();
        private final AtomicLong putBlockOperations = new AtomicLong();
        private final AtomicLong putBlockListOperations = new AtomicLong();

        private Stats() {
        }

        private Map<String, Long> toMap() {
            return Map.of("GetBlob", this.getOperations.get(), "ListBlobs", this.listOperations.get(), "GetBlobProperties", this.headOperations.get(), "PutBlob", this.putOperations.get(), "PutBlock", this.putBlockOperations.get(), "PutBlockList", this.putBlockListOperations.get());
        }
    }

    private static class RequestStatsCollector {
        private final BiPredicate<String, URL> filter;
        private final Runnable onHttpRequest;

        private RequestStatsCollector(BiPredicate<String, URL> filter, Runnable onHttpRequest) {
            this.filter = filter;
            this.onHttpRequest = onHttpRequest;
        }

        static RequestStatsCollector create(BiPredicate<String, URL> filter, Runnable consumer) {
            return new RequestStatsCollector(filter, consumer);
        }

        private boolean shouldConsumeRequestInfo(String httpMethod, URL url) {
            return this.filter.test(httpMethod, url);
        }

        private void consumeHttpRequestInfo() {
            this.onHttpRequest.run();
        }
    }

    private static class AzureInputStream
    extends InputStream {
        private final CancellableRateLimitedFluxIterator<ByteBuf> cancellableRateLimitedFluxIterator;
        private ByteBuf byteBuf;
        private boolean closed;
        private final ByteBufAllocator allocator;

        private AzureInputStream(BlobAsyncClient client, long rangeOffset, long rangeLength, long contentLength, int maxRetries, ByteBufAllocator allocator) throws IOException {
            rangeLength = Math.min(rangeLength, contentLength - rangeOffset);
            BlobRange range = new BlobRange(rangeOffset, Long.valueOf(rangeLength));
            DownloadRetryOptions downloadRetryOptions = new DownloadRetryOptions().setMaxRetryRequests(maxRetries);
            Flux byteBufFlux = client.downloadWithResponse(range, downloadRetryOptions, null, false).flux().concatMap(ResponseBase::getValue).filter(Objects::nonNull).map(this::copyBuffer);
            this.allocator = allocator;
            this.cancellableRateLimitedFluxIterator = new CancellableRateLimitedFluxIterator<ByteBuf>(8, ReferenceCountUtil::safeRelease);
            byteBufFlux.subscribe(this.cancellableRateLimitedFluxIterator);
            this.getNextByteBuf();
        }

        private ByteBuf copyBuffer(ByteBuffer buffer) {
            ByteBuf byteBuffer = this.allocator.heapBuffer(buffer.remaining(), buffer.remaining());
            byteBuffer.writeBytes(buffer);
            return byteBuffer;
        }

        @Override
        public int read() throws IOException {
            byte[] b = new byte[1];
            int bytesRead = this.read(b, 0, 1);
            if (bytesRead > 1) {
                throw new IOException("Stream returned more data than requested");
            }
            if (bytesRead == 1) {
                return b[0] & 0xFF;
            }
            if (bytesRead == 0) {
                throw new IOException("Stream returned unexpected number of bytes");
            }
            return -1;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            int totalBytesRead;
            int toRead;
            if (off < 0 || len < 0 || len > b.length - off) {
                throw new IndexOutOfBoundsException();
            }
            ByteBuf buffer = this.getNextByteBuf();
            if (buffer == null || buffer.readableBytes() == 0) {
                this.releaseByteBuf(buffer);
                return -1;
            }
            for (totalBytesRead = 0; buffer != null && totalBytesRead < len; totalBytesRead += toRead) {
                toRead = Math.min(len - totalBytesRead, buffer.readableBytes());
                buffer.readBytes(b, off + totalBytesRead, toRead);
                if (buffer.readableBytes() != 0) continue;
                this.releaseByteBuf(buffer);
                buffer = this.getNextByteBuf();
            }
            return totalBytesRead;
        }

        @Override
        public void close() {
            if (!this.closed) {
                this.cancellableRateLimitedFluxIterator.cancel();
                this.closed = true;
                this.releaseByteBuf(this.byteBuf);
            }
        }

        private void releaseByteBuf(ByteBuf buf) {
            ReferenceCountUtil.safeRelease((Object)buf);
            this.byteBuf = null;
        }

        @Nullable
        private ByteBuf getNextByteBuf() throws IOException {
            try {
                if (this.byteBuf == null && !this.cancellableRateLimitedFluxIterator.hasNext()) {
                    return null;
                }
                if (this.byteBuf != null) {
                    return this.byteBuf;
                }
                this.byteBuf = this.cancellableRateLimitedFluxIterator.next();
                return this.byteBuf;
            }
            catch (Exception e) {
                throw new IOException("Unable to read blob", e.getCause());
            }
        }
    }
}

