/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.translog;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.LongConsumer;
import java.util.function.LongSupplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.lucene.store.AlreadyClosedException;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.io.DiskIoBufferPool;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.ReleasableBytesStreamOutput;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.common.util.concurrent.ReleasableLock;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Releasables;
import org.elasticsearch.core.Strings;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.mapper.Uid;
import org.elasticsearch.index.shard.AbstractIndexShardComponent;
import org.elasticsearch.index.shard.IndexShardComponent;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.translog.BaseTranslogReader;
import org.elasticsearch.index.translog.BufferedChecksumStreamInput;
import org.elasticsearch.index.translog.BufferedChecksumStreamOutput;
import org.elasticsearch.index.translog.ChannelFactory;
import org.elasticsearch.index.translog.Checkpoint;
import org.elasticsearch.index.translog.MultiSnapshot;
import org.elasticsearch.index.translog.OperationListener;
import org.elasticsearch.index.translog.TragicExceptionHolder;
import org.elasticsearch.index.translog.TranslogConfig;
import org.elasticsearch.index.translog.TranslogCorruptedException;
import org.elasticsearch.index.translog.TranslogDeletionPolicy;
import org.elasticsearch.index.translog.TranslogException;
import org.elasticsearch.index.translog.TranslogHeader;
import org.elasticsearch.index.translog.TranslogReader;
import org.elasticsearch.index.translog.TranslogSnapshot;
import org.elasticsearch.index.translog.TranslogStats;
import org.elasticsearch.index.translog.TranslogWriter;
import org.elasticsearch.index.translog.TruncatedTranslogException;

public class Translog
extends AbstractIndexShardComponent
implements IndexShardComponent,
Closeable {
    public static final String TRANSLOG_UUID_KEY = "translog_uuid";
    public static final String TRANSLOG_FILE_PREFIX = "translog-";
    public static final String TRANSLOG_FILE_SUFFIX = ".tlog";
    public static final String CHECKPOINT_SUFFIX = ".ckp";
    public static final String CHECKPOINT_FILE_NAME = "translog.ckp";
    static final Pattern PARSE_STRICT_ID_PATTERN = Pattern.compile("^translog-(\\d+)(\\.tlog)$");
    public static final int DEFAULT_HEADER_SIZE_IN_BYTES = TranslogHeader.headerSizeInBytes(UUIDs.randomBase64UUID());
    private final List<TranslogReader> readers = new ArrayList<TranslogReader>();
    private final BigArrays bigArrays;
    private final DiskIoBufferPool diskIoBufferPool;
    protected final ReleasableLock readLock;
    protected final ReleasableLock writeLock;
    private final Path location;
    private TranslogWriter current;
    protected final TragicExceptionHolder tragedy = new TragicExceptionHolder();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final TranslogConfig config;
    private final LongSupplier globalCheckpointSupplier;
    private final LongSupplier primaryTermSupplier;
    private final String translogUUID;
    private final TranslogDeletionPolicy deletionPolicy;
    private final LongConsumer persistedSequenceNumberConsumer;
    private final OperationListener operationListener;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Translog(TranslogConfig config, String translogUUID, TranslogDeletionPolicy deletionPolicy, LongSupplier globalCheckpointSupplier, LongSupplier primaryTermSupplier, LongConsumer persistedSequenceNumberConsumer) throws IOException {
        super(config.getShardId(), config.getIndexSettings());
        this.config = config;
        this.globalCheckpointSupplier = globalCheckpointSupplier;
        this.primaryTermSupplier = primaryTermSupplier;
        this.persistedSequenceNumberConsumer = persistedSequenceNumberConsumer;
        this.operationListener = config.getOperationListener();
        this.deletionPolicy = deletionPolicy;
        this.translogUUID = translogUUID;
        this.bigArrays = config.getBigArrays();
        this.diskIoBufferPool = config.getDiskIoBufferPool();
        ReentrantReadWriteLock rwl = new ReentrantReadWriteLock();
        this.readLock = new ReleasableLock(rwl.readLock());
        this.writeLock = new ReleasableLock(rwl.writeLock());
        this.location = config.getTranslogPath();
        Files.createDirectories(this.location, new FileAttribute[0]);
        try {
            Checkpoint checkpoint = Translog.readCheckpoint(this.location);
            Path nextTranslogFile = this.location.resolve(Translog.getFilename(checkpoint.generation + 1L));
            Path currentCheckpointFile = this.location.resolve(Translog.getCommitCheckpointFileName(checkpoint.generation));
            assert (!Files.exists(nextTranslogFile, new LinkOption[0]) || Files.size(nextTranslogFile) <= (long)TranslogHeader.headerSizeInBytes(translogUUID)) : "unexpected translog file: [" + nextTranslogFile + "]";
            if (Files.exists(currentCheckpointFile, new LinkOption[0]) && Files.deleteIfExists(nextTranslogFile)) {
                this.logger.warn("deleted previously created, but not yet committed, next generation [{}]. This can happen due to a tragic exception when creating a new generation", (Object)nextTranslogFile.getFileName());
            }
            this.readers.addAll(this.recoverFromFiles(checkpoint));
            if (this.readers.isEmpty()) {
                throw new IllegalStateException("at least one reader must be recovered");
            }
            boolean success = false;
            this.current = null;
            try {
                this.current = this.createWriter(checkpoint.generation + 1L, this.getMinFileGeneration(), checkpoint.globalCheckpoint, persistedSequenceNumberConsumer);
                success = true;
            }
            finally {
                if (!success) {
                    IOUtils.closeWhileHandlingException(this.readers);
                }
            }
        }
        catch (Exception e) {
            IOUtils.closeWhileHandlingException((Closeable)this.current);
            IOUtils.closeWhileHandlingException(this.readers);
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ArrayList<TranslogReader> recoverFromFiles(Checkpoint checkpoint) throws IOException {
        boolean success = false;
        ArrayList<TranslogReader> foundTranslogs = new ArrayList<TranslogReader>();
        try (ReleasableLock ignored = this.writeLock.acquire();){
            this.logger.debug("open uncommitted translog checkpoint {}", (Object)checkpoint);
            long minGenerationToRecoverFrom = checkpoint.minTranslogGeneration;
            assert (minGenerationToRecoverFrom >= 0L) : "minTranslogGeneration should be non-negative";
            for (long i = checkpoint.generation; i >= minGenerationToRecoverFrom; --i) {
                TranslogReader reader;
                Path committedTranslogFile = this.location.resolve(Translog.getFilename(i));
                Checkpoint readerCheckpoint = i == checkpoint.generation ? checkpoint : Checkpoint.read(this.location.resolve(Translog.getCommitCheckpointFileName(i)));
                try {
                    reader = this.openReader(committedTranslogFile, readerCheckpoint);
                }
                catch (NoSuchFileException fnfe) {
                    throw new TranslogCorruptedException(committedTranslogFile.toString(), "translog file doesn't exist with generation: " + i + " recovering from: " + minGenerationToRecoverFrom + " checkpoint: " + checkpoint.generation + " - translog ids must be consecutive");
                }
                assert (reader.getPrimaryTerm() <= this.primaryTermSupplier.getAsLong()) : "Primary terms go backwards; current term [" + this.primaryTermSupplier.getAsLong() + "] translog path [ " + committedTranslogFile + ", existing term [" + reader.getPrimaryTerm() + "]";
                foundTranslogs.add(reader);
                this.logger.debug("recovered local translog from checkpoint {}", (Object)checkpoint);
            }
            Collections.reverse(foundTranslogs);
            IOUtils.deleteFilesIgnoringExceptions((Path[])new Path[]{this.location.resolve(Translog.getFilename(minGenerationToRecoverFrom - 1L)), this.location.resolve(Translog.getCommitCheckpointFileName(minGenerationToRecoverFrom - 1L))});
            Path commitCheckpoint = this.location.resolve(Translog.getCommitCheckpointFileName(checkpoint.generation));
            if (Files.exists(commitCheckpoint, new LinkOption[0])) {
                Checkpoint checkpointFromDisk = Checkpoint.read(commitCheckpoint);
                if (!checkpoint.equals(checkpointFromDisk)) {
                    throw new TranslogCorruptedException(commitCheckpoint.toString(), "checkpoint file " + commitCheckpoint.getFileName() + " already exists but has corrupted content: expected " + checkpoint + " but got " + checkpointFromDisk);
                }
            } else {
                this.copyCheckpointTo(commitCheckpoint);
            }
            success = true;
        }
        finally {
            if (!success) {
                IOUtils.closeWhileHandlingException(foundTranslogs);
            }
        }
        return foundTranslogs;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyCheckpointTo(Path targetPath) throws IOException {
        Path tempFile = Files.createTempFile(this.location, TRANSLOG_FILE_PREFIX, CHECKPOINT_SUFFIX, new FileAttribute[0]);
        boolean tempFileRenamed = false;
        try {
            Files.copy(this.location.resolve(CHECKPOINT_FILE_NAME), tempFile, StandardCopyOption.REPLACE_EXISTING);
            IOUtils.fsync((Path)tempFile, (boolean)false);
            Files.move(tempFile, targetPath, StandardCopyOption.ATOMIC_MOVE);
            tempFileRenamed = true;
            IOUtils.fsync((Path)targetPath.getParent(), (boolean)true);
        }
        finally {
            if (!tempFileRenamed) {
                try {
                    Files.delete(tempFile);
                }
                catch (IOException ex) {
                    this.logger.warn(() -> Strings.format((String)"failed to delete temp file %s", (Object[])new Object[]{tempFile}), (Throwable)ex);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    TranslogReader openReader(Path path, Checkpoint checkpoint) throws IOException {
        FileChannel channel = FileChannel.open(path, StandardOpenOption.READ);
        try {
            assert (Translog.parseIdFromFileName(path) == checkpoint.generation) : "expected generation: " + Translog.parseIdFromFileName(path) + " but got: " + checkpoint.generation;
            TranslogReader reader = TranslogReader.open(channel, path, checkpoint, this.translogUUID);
            channel = null;
            TranslogReader translogReader = reader;
            return translogReader;
        }
        finally {
            IOUtils.close((Closeable)channel);
        }
    }

    public static long parseIdFromFileName(Path translogFile) {
        String fileName = translogFile.getFileName().toString();
        Matcher matcher = PARSE_STRICT_ID_PATTERN.matcher(fileName);
        if (matcher.matches()) {
            try {
                return Long.parseLong(matcher.group(1));
            }
            catch (NumberFormatException e) {
                throw new IllegalStateException("number formatting issue in a file that passed PARSE_STRICT_ID_PATTERN: " + fileName + "]", e);
            }
        }
        throw new IllegalArgumentException("can't parse id from file: " + fileName);
    }

    public boolean isOpen() {
        return !this.closed.get();
    }

    private static boolean calledFromOutsideOrViaTragedyClose() {
        List<StackTraceElement> frames = Stream.of(Thread.currentThread().getStackTrace()).skip(3L).limit(10L).filter(f -> {
            try {
                return Translog.class.isAssignableFrom(Class.forName(f.getClassName()));
            }
            catch (Exception ignored) {
                return false;
            }
        }).toList();
        return frames.isEmpty() || frames.stream().anyMatch(f -> f.getMethodName().equals("closeOnTragicEvent"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        assert (Translog.calledFromOutsideOrViaTragedyClose()) : "Translog.close method is called from inside Translog, but not via closeOnTragicEvent method";
        if (this.closed.compareAndSet(false, true)) {
            try (ReleasableLock lock = this.writeLock.acquire();){
                try {
                    this.current.sync();
                }
                finally {
                    this.closeFilesIfNoPendingRetentionLocks();
                }
            }
            finally {
                this.logger.debug("translog closed");
            }
        }
    }

    public Path location() {
        return this.location;
    }

    public long currentFileGeneration() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            long l = this.current.getGeneration();
            return l;
        }
    }

    public long getMinFileGeneration() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            if (this.readers.isEmpty()) {
                long l = this.current.getGeneration();
                return l;
            }
            assert (this.readers.stream().map(BaseTranslogReader::getGeneration).min(Long::compareTo).get().equals(this.readers.get(0).getGeneration())) : "the first translog isn't the one with the minimum generation:" + this.readers;
            long l = this.readers.get(0).getGeneration();
            return l;
        }
    }

    public int totalOperations() {
        return this.totalOperationsByMinGen(-1L);
    }

    public long sizeInBytes() {
        return this.sizeInBytesByMinGen(-1L);
    }

    long earliestLastModifiedAge() {
        ReleasableLock ignored = this.readLock.acquire();
        try {
            this.ensureOpen();
            long l = Translog.findEarliestLastModifiedAge(System.currentTimeMillis(), this.readers, this.current);
            if (ignored != null) {
                ignored.close();
            }
            return l;
        }
        catch (Throwable throwable) {
            try {
                if (ignored != null) {
                    try {
                        ignored.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new TranslogException(this.shardId, "Unable to get the earliest last modified time for the transaction log");
            }
        }
    }

    static long findEarliestLastModifiedAge(long currentTime, Iterable<TranslogReader> readers, TranslogWriter writer) throws IOException {
        long earliestTime = currentTime;
        for (TranslogReader r : readers) {
            earliestTime = Math.min(r.getLastModifiedTime(), earliestTime);
        }
        return Math.max(0L, currentTime - Math.min(earliestTime, writer.getLastModifiedTime()));
    }

    public int totalOperationsByMinGen(long minGeneration) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            int n = Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(r -> r.getGeneration() >= minGeneration).mapToInt(rec$ -> ((BaseTranslogReader)rec$).totalOperations()).sum();
            return n;
        }
    }

    public int estimateTotalOperationsFromMinSeq(long minSeqNo) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            int n = this.readersAboveMinSeqNo(minSeqNo).mapToInt(BaseTranslogReader::totalOperations).sum();
            return n;
        }
    }

    public long sizeInBytesByMinGen(long minGeneration) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            long l = Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(r -> r.getGeneration() >= minGeneration).mapToLong(rec$ -> ((BaseTranslogReader)rec$).sizeInBytes()).sum();
            return l;
        }
    }

    TranslogWriter createWriter(long fileGeneration) throws IOException {
        TranslogWriter writer = this.createWriter(fileGeneration, this.getMinFileGeneration(), this.globalCheckpointSupplier.getAsLong(), this.persistedSequenceNumberConsumer);
        assert (writer.sizeInBytes() == (long)DEFAULT_HEADER_SIZE_IN_BYTES) : "Mismatch translog header size; empty translog size [" + writer.sizeInBytes() + ", header size [" + DEFAULT_HEADER_SIZE_IN_BYTES + "]";
        return writer;
    }

    TranslogWriter createWriter(long fileGeneration, long initialMinTranslogGen, long initialGlobalCheckpoint, LongConsumer persistedSequenceNumberConsumer) throws IOException {
        TranslogWriter newWriter;
        try {
            newWriter = TranslogWriter.create(this.shardId, this.translogUUID, fileGeneration, this.location.resolve(Translog.getFilename(fileGeneration)), this.getChannelFactory(), this.config.getBufferSize(), initialMinTranslogGen, initialGlobalCheckpoint, this.globalCheckpointSupplier, this::getMinFileGeneration, this.primaryTermSupplier.getAsLong(), this.tragedy, persistedSequenceNumberConsumer, this.bigArrays, this.diskIoBufferPool, this.operationListener);
        }
        catch (IOException e) {
            throw new TranslogException(this.shardId, "failed to create new translog file", e);
        }
        return newWriter;
    }

    public Location add(Operation operation) throws IOException {
        ReleasableBytesStreamOutput out = new ReleasableBytesStreamOutput(this.bigArrays);
        try {
            Translog.writeOperationWithSize(out, operation);
            BytesReference bytes = out.bytes();
            ReleasableLock ignored = this.readLock.acquire();
            try {
                this.ensureOpen();
                if (operation.primaryTerm() > this.current.getPrimaryTerm()) {
                    assert (false) : "Operation term is newer than the current term; current term[" + this.current.getPrimaryTerm() + "], operation term[" + operation + "]";
                    throw new IllegalArgumentException("Operation term is newer than the current term; current term[" + this.current.getPrimaryTerm() + "], operation term[" + operation + "]");
                }
                Location location = this.current.add(bytes, operation.seqNo());
                if (ignored != null) {
                    ignored.close();
                }
                return location;
            }
            catch (Throwable throwable) {
                try {
                    if (ignored != null) {
                        try {
                            ignored.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException | AlreadyClosedException ex) {
                    this.closeOnTragicEvent((Exception)ex);
                    throw ex;
                }
                catch (Exception ex) {
                    this.closeOnTragicEvent(ex);
                    throw new TranslogException(this.shardId, "Failed to write operation [" + operation + "]", ex);
                }
            }
        }
        finally {
            Releasables.close((Releasable)out);
        }
    }

    public boolean shouldRollGeneration() {
        long threshold = this.indexSettings.getGenerationThresholdSize().getBytes();
        try (ReleasableLock ignored = this.readLock.acquire();){
            boolean bl = this.current.sizeInBytes() > threshold;
            return bl;
        }
    }

    public Location getLastWriteLocation() {
        try (ReleasableLock lock = this.readLock.acquire();){
            Location location = new Location(this.current.generation, this.current.sizeInBytes() - 1L, Integer.MAX_VALUE);
            return location;
        }
    }

    public long getLastSyncedGlobalCheckpoint() {
        return this.getLastSyncedCheckpoint().globalCheckpoint;
    }

    final Checkpoint getLastSyncedCheckpoint() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            Checkpoint checkpoint = this.current.getLastSyncedCheckpoint();
            return checkpoint;
        }
    }

    public Snapshot newSnapshot() throws IOException {
        return this.newSnapshot(0L, Long.MAX_VALUE);
    }

    public Snapshot newSnapshot(long fromSeqNo, long toSeqNo) throws IOException {
        assert (fromSeqNo <= toSeqNo) : fromSeqNo + " > " + toSeqNo;
        assert (fromSeqNo >= 0L) : "from_seq_no must be non-negative " + fromSeqNo;
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            TranslogSnapshot[] snapshots = (TranslogSnapshot[])Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(reader -> reader.getCheckpoint().minSeqNo <= toSeqNo && fromSeqNo <= reader.getCheckpoint().maxEffectiveSeqNo()).map(rec$ -> ((BaseTranslogReader)rec$).newSnapshot()).toArray(TranslogSnapshot[]::new);
            Snapshot snapshot = this.newMultiSnapshot(snapshots);
            SeqNoFilterSnapshot seqNoFilterSnapshot = new SeqNoFilterSnapshot(snapshot, fromSeqNo, toSeqNo);
            return seqNoFilterSnapshot;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Operation readOperation(Location location) throws IOException {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            if (location.generation < this.getMinFileGeneration()) {
                Operation operation = null;
                return operation;
            }
            if (this.current.generation == location.generation) {
                Operation operation = this.current.read(location);
                return operation;
            }
            int i = this.readers.size() - 1;
            while (i >= 0) {
                TranslogReader translogReader = this.readers.get(i);
                if (translogReader.generation == location.generation) {
                    Operation operation = translogReader.read(location);
                    return operation;
                }
                --i;
            }
            return null;
        }
        catch (Exception ex) {
            this.closeOnTragicEvent(ex);
            throw ex;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Snapshot newMultiSnapshot(TranslogSnapshot[] snapshots) throws IOException {
        Closeable onClose;
        if (snapshots.length == 0) {
            onClose = () -> {};
        } else {
            assert (Arrays.stream(snapshots).map(BaseTranslogReader::getGeneration).min(Long::compareTo).get() == snapshots[0].generation) : "first reader generation of " + snapshots + " is not the smallest";
            onClose = this.acquireTranslogGenFromDeletionPolicy(snapshots[0].generation);
        }
        boolean success = false;
        try {
            MultiSnapshot result = new MultiSnapshot(snapshots, onClose);
            success = true;
            MultiSnapshot multiSnapshot = result;
            return multiSnapshot;
        }
        finally {
            if (!success) {
                onClose.close();
            }
        }
    }

    private Stream<? extends BaseTranslogReader> readersAboveMinSeqNo(long minSeqNo) {
        assert (this.readLock.isHeldByCurrentThread() || this.writeLock.isHeldByCurrentThread()) : "callers of readersAboveMinSeqNo must hold a lock: readLock [" + this.readLock.isHeldByCurrentThread() + "], writeLock [" + this.readLock.isHeldByCurrentThread() + "]";
        return Stream.concat(this.readers.stream(), Stream.of(this.current)).filter(reader -> minSeqNo <= reader.getCheckpoint().maxEffectiveSeqNo());
    }

    public Closeable acquireRetentionLock() {
        try (ReleasableLock lock = this.readLock.acquire();){
            this.ensureOpen();
            long viewGen = this.getMinFileGeneration();
            Closeable closeable = this.acquireTranslogGenFromDeletionPolicy(viewGen);
            return closeable;
        }
    }

    private Closeable acquireTranslogGenFromDeletionPolicy(long viewGen) {
        Releasable toClose = this.deletionPolicy.acquireTranslogGen(viewGen);
        return () -> {
            try {
                toClose.close();
            }
            finally {
                this.trimUnreferencedReaders();
                this.closeFilesIfNoPendingRetentionLocks();
            }
        };
    }

    public void sync() throws IOException {
        try (ReleasableLock lock = this.readLock.acquire();){
            if (!this.closed.get()) {
                this.current.sync();
            }
        }
        catch (Exception ex) {
            this.closeOnTragicEvent(ex);
            throw ex;
        }
    }

    public boolean syncNeeded() {
        try (ReleasableLock lock = this.readLock.acquire();){
            boolean bl = this.current.syncNeeded();
            return bl;
        }
    }

    public static String getFilename(long generation) {
        return TRANSLOG_FILE_PREFIX + generation + TRANSLOG_FILE_SUFFIX;
    }

    static String getCommitCheckpointFileName(long generation) {
        return TRANSLOG_FILE_PREFIX + generation + CHECKPOINT_SUFFIX;
    }

    public void trimOperations(long belowTerm, long aboveSeqNo) throws IOException {
        assert (aboveSeqNo >= -1L) : "aboveSeqNo has to a valid sequence number";
        try (ReleasableLock lock = this.writeLock.acquire();){
            this.ensureOpen();
            if (this.current.getPrimaryTerm() < belowTerm) {
                throw new IllegalArgumentException("Trimming the translog can only be done for terms lower than the current one. Trim requested for term [ " + belowTerm + " ] , current is [ " + this.current.getPrimaryTerm() + " ]");
            }
            assert (this.current.assertNoSeqAbove(belowTerm, aboveSeqNo));
            ArrayList<TranslogReader> newReaders = new ArrayList<TranslogReader>(this.readers.size());
            try {
                for (TranslogReader reader : this.readers) {
                    TranslogReader newReader = reader.getPrimaryTerm() < belowTerm ? reader.closeIntoTrimmedReader(aboveSeqNo, this.getChannelFactory()) : reader;
                    newReaders.add(newReader);
                }
            }
            catch (IOException e) {
                IOUtils.closeWhileHandlingException(newReaders);
                this.tragedy.setTragicException(e);
                this.closeOnTragicEvent(e);
                throw e;
            }
            this.readers.clear();
            this.readers.addAll(newReaders);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean ensureSynced(Location location, long globalCheckpoint) throws IOException {
        try (ReleasableLock lock = this.readLock.acquire();){
            long persistedGlobalCheckpoint = this.current.getLastSyncedCheckpoint().globalCheckpoint;
            if (location.generation != this.current.getGeneration()) {
                if (persistedGlobalCheckpoint >= globalCheckpoint) return false;
            }
            this.ensureOpen();
            boolean bl = this.current.syncUpTo(location.translogLocation + (long)location.size, globalCheckpoint);
            return bl;
        }
        catch (Exception ex) {
            this.closeOnTragicEvent(ex);
            throw ex;
        }
    }

    protected void closeOnTragicEvent(Exception ex) {
        assert (!this.readLock.isHeldByCurrentThread()) : Thread.currentThread().getName();
        if (this.tragedy.get() != null) {
            try {
                this.close();
            }
            catch (AlreadyClosedException alreadyClosedException) {
            }
            catch (Exception inner) {
                assert (ex != inner.getCause());
                ex.addSuppressed(inner);
            }
        }
    }

    public TranslogStats stats() {
        try (ReleasableLock lock = this.readLock.acquire();){
            long uncommittedGen = Translog.minGenerationForSeqNo(this.deletionPolicy.getLocalCheckpointOfSafeCommit() + 1L, this.current, this.readers);
            TranslogStats translogStats = new TranslogStats(this.totalOperations(), this.sizeInBytes(), this.totalOperationsByMinGen(uncommittedGen), this.sizeInBytesByMinGen(uncommittedGen), this.earliestLastModifiedAge());
            return translogStats;
        }
    }

    public TranslogConfig getConfig() {
        return this.config;
    }

    public TranslogDeletionPolicy getDeletionPolicy() {
        return this.deletionPolicy;
    }

    static void verifyChecksum(BufferedChecksumStreamInput in) throws IOException {
        long expectedChecksum = in.getChecksum();
        long readChecksum = Integer.toUnsignedLong(in.readInt());
        if (readChecksum != expectedChecksum) {
            throw new TranslogCorruptedException(in.getSource(), "checksum verification failed - expected: 0x" + Long.toHexString(expectedChecksum) + ", got: 0x" + Long.toHexString(readChecksum));
        }
    }

    public static List<Operation> readOperations(StreamInput input, String source) throws IOException {
        ArrayList<Operation> operations = new ArrayList<Operation>();
        int numOps = input.readInt();
        BufferedChecksumStreamInput checksumStreamInput = new BufferedChecksumStreamInput(input, source);
        if (input.getTransportVersion().before(TransportVersion.V_8_8_0)) {
            for (int i = 0; i < numOps; ++i) {
                operations.add(Translog.readOperation(checksumStreamInput));
            }
        } else {
            for (int i = 0; i < numOps; ++i) {
                checksumStreamInput.resetDigest();
                operations.add(Operation.readOperation(checksumStreamInput));
                Translog.verifyChecksum(checksumStreamInput);
            }
        }
        return operations;
    }

    public static Operation readOperation(BufferedChecksumStreamInput in) throws IOException {
        Operation operation;
        try {
            int opSize = in.readInt();
            if (opSize < 4) {
                throw new TranslogCorruptedException(in.getSource(), "operation size must be at least 4 but was: " + opSize);
            }
            in.resetDigest();
            if (in.markSupported()) {
                in.mark(opSize);
                in.skip(opSize - 4);
                Translog.verifyChecksum(in);
                in.reset();
            }
            operation = Operation.readOperation(in);
            Translog.verifyChecksum(in);
        }
        catch (EOFException e) {
            throw new TruncatedTranslogException(in.getSource(), "reached premature end of file, translog is truncated", e);
        }
        return operation;
    }

    public static void writeOperations(StreamOutput outStream, List<Operation> toWrite) throws IOException {
        int size = toWrite.size();
        outStream.writeInt(size);
        if (size == 0) {
            return;
        }
        if (outStream.getTransportVersion().onOrAfter(TransportVersion.V_8_8_0)) {
            BufferedChecksumStreamOutput checksumStreamOutput = new BufferedChecksumStreamOutput(outStream);
            for (Operation op : toWrite) {
                Translog.writeOperationNoSize(checksumStreamOutput, op);
            }
        } else {
            Translog.writeOperationsToStreamLegacyFormat(outStream, toWrite);
        }
    }

    private static void writeOperationsToStreamLegacyFormat(StreamOutput outStream, List<Operation> toWrite) throws IOException {
        BytesStreamOutput out = new BytesStreamOutput();
        BufferedChecksumStreamOutput checksumStreamOutput = new BufferedChecksumStreamOutput(out);
        for (Operation op : toWrite) {
            out.reset();
            Translog.writeOperationNoSize(checksumStreamOutput, op);
            outStream.writeInt(Math.toIntExact(out.position()));
            out.bytes().writeTo(outStream);
        }
    }

    public static void writeOperationNoSize(BufferedChecksumStreamOutput out, Operation op) throws IOException {
        out.resetDigest();
        op.writeTo(out);
        long checksum = out.getChecksum();
        out.writeInt((int)checksum);
    }

    public static void writeOperationWithSize(BytesStreamOutput out, Operation op) throws IOException {
        long start = out.position();
        out.skip(4);
        Translog.writeOperationNoSize(new BufferedChecksumStreamOutput(out), op);
        long end = out.position();
        int operationSize = (int)(end - 4L - start);
        out.seek(start);
        out.writeInt(operationSize);
        out.seek(end);
    }

    public TranslogGeneration getMinGenerationForSeqNo(long seqNo) {
        try (ReleasableLock ignored = this.readLock.acquire();){
            TranslogGeneration translogGeneration = new TranslogGeneration(this.translogUUID, Translog.minGenerationForSeqNo(seqNo, this.current, this.readers));
            return translogGeneration;
        }
    }

    private static long minGenerationForSeqNo(long seqNo, TranslogWriter writer, List<TranslogReader> readers) {
        long minGen = writer.generation;
        for (TranslogReader reader : readers) {
            if (seqNo > reader.getCheckpoint().maxEffectiveSeqNo()) continue;
            minGen = Math.min(minGen, reader.getGeneration());
        }
        return minGen;
    }

    public void rollGeneration() throws IOException {
        this.syncBeforeRollGeneration();
        if (this.current.totalOperations() == 0 && this.primaryTermSupplier.getAsLong() == this.current.getPrimaryTerm()) {
            return;
        }
        try (ReleasableLock ignored = this.writeLock.acquire();){
            this.ensureOpen();
            try {
                TranslogReader reader = this.current.closeIntoReader();
                this.readers.add(reader);
                assert (Checkpoint.read((Path)this.location.resolve((String)CHECKPOINT_FILE_NAME)).generation == this.current.getGeneration());
                this.copyCheckpointTo(this.location.resolve(Translog.getCommitCheckpointFileName(this.current.getGeneration())));
                this.current = this.createWriter(this.current.getGeneration() + 1L);
                this.logger.trace("current translog set to [{}]", (Object)this.current.getGeneration());
            }
            catch (Exception e) {
                this.tragedy.setTragicException(e);
                this.closeOnTragicEvent(e);
                throw e;
            }
        }
    }

    void syncBeforeRollGeneration() throws IOException {
        this.sync();
    }

    public void trimUnreferencedReaders() throws IOException {
        try (ReleasableLock ignored = this.readLock.acquire();){
            if (this.closed.get()) {
                return;
            }
            if (this.getMinReferencedGen() == this.getMinFileGeneration()) {
                return;
            }
        }
        this.sync();
        try {
            ignored = this.writeLock.acquire();
            try {
                TranslogReader reader;
                if (this.closed.get()) {
                    return;
                }
                long minReferencedGen = this.getMinReferencedGen();
                Iterator<TranslogReader> iterator = this.readers.iterator();
                while (iterator.hasNext() && (reader = iterator.next()).getGeneration() < minReferencedGen) {
                    iterator.remove();
                    IOUtils.closeWhileHandlingException((Closeable)reader);
                    Path translogPath = reader.path();
                    this.logger.trace("delete translog file [{}], not referenced and not current anymore", (Object)translogPath);
                    this.current.sync();
                    this.deleteReaderFiles(reader);
                }
                assert (!this.readers.isEmpty() || this.current.generation == minReferencedGen) : "all readers were cleaned but the minReferenceGen [" + minReferencedGen + "] is not the current writer's gen [" + this.current.generation + "]";
            }
            finally {
                if (ignored != null) {
                    ignored.close();
                }
            }
        }
        catch (Exception ex) {
            this.closeOnTragicEvent(ex);
            throw ex;
        }
    }

    private long getMinReferencedGen() {
        assert (this.readLock.isHeldByCurrentThread() || this.writeLock.isHeldByCurrentThread());
        long minReferencedGen = Math.min(this.deletionPolicy.getMinTranslogGenRequiredByLocks(), Translog.minGenerationForSeqNo(this.deletionPolicy.getLocalCheckpointOfSafeCommit() + 1L, this.current, this.readers));
        assert (minReferencedGen >= this.getMinFileGeneration()) : "deletion policy requires a minReferenceGen of [" + minReferencedGen + "] but the lowest gen available is [" + this.getMinFileGeneration() + "]";
        assert (minReferencedGen <= this.currentFileGeneration()) : "deletion policy requires a minReferenceGen of [" + minReferencedGen + "] which is higher than the current generation [" + this.currentFileGeneration() + "]";
        return minReferencedGen;
    }

    void deleteReaderFiles(TranslogReader reader) {
        IOUtils.deleteFilesIgnoringExceptions((Path[])new Path[]{reader.path(), reader.path().resolveSibling(Translog.getCommitCheckpointFileName(reader.getGeneration()))});
    }

    void closeFilesIfNoPendingRetentionLocks() throws IOException {
        try (ReleasableLock ignored = this.writeLock.acquire();){
            if (this.closed.get() && this.deletionPolicy.pendingTranslogRefCount() == 0) {
                this.logger.trace("closing files. translog is closed and there are no pending retention locks");
                ArrayList<TranslogReader> toClose = new ArrayList<TranslogReader>(this.readers);
                toClose.add((TranslogReader)((Object)this.current));
                IOUtils.close(toClose);
            }
        }
    }

    public TranslogGeneration getGeneration() {
        return new TranslogGeneration(this.translogUUID, this.currentFileGeneration());
    }

    long getFirstOperationPosition() {
        return this.current.getFirstOperationOffset();
    }

    private void ensureOpen() {
        if (this.closed.get()) {
            throw new AlreadyClosedException("translog is already closed", (Throwable)this.tragedy.get());
        }
    }

    ChannelFactory getChannelFactory() {
        return FileChannel::open;
    }

    public Exception getTragicException() {
        return this.tragedy.get();
    }

    static Checkpoint readCheckpoint(Path location) throws IOException {
        return Checkpoint.read(location.resolve(CHECKPOINT_FILE_NAME));
    }

    public static long readGlobalCheckpoint(Path location, String expectedTranslogUUID) throws IOException {
        Checkpoint checkpoint = Translog.readCheckpoint(location, expectedTranslogUUID);
        return checkpoint.globalCheckpoint;
    }

    private static Checkpoint readCheckpoint(Path location, String expectedTranslogUUID) throws IOException {
        Checkpoint checkpoint = Translog.readCheckpoint(location);
        Path translogFile = location.resolve(Translog.getFilename(checkpoint.generation));
        try (FileChannel channel = FileChannel.open(translogFile, StandardOpenOption.READ);){
            TranslogHeader.read(expectedTranslogUUID, translogFile, channel);
        }
        catch (TranslogCorruptedException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new TranslogCorruptedException(location.toString(), ex);
        }
        return checkpoint;
    }

    public String getTranslogUUID() {
        return this.translogUUID;
    }

    public long getMaxSeqNo() {
        try (ReleasableLock ignored = this.readLock.acquire();){
            this.ensureOpen();
            OptionalLong maxSeqNo = Stream.concat(this.readers.stream(), Stream.of(this.current)).mapToLong(reader -> reader.getCheckpoint().maxSeqNo).max();
            assert (maxSeqNo.isPresent()) : "must have at least one translog generation";
            long l = maxSeqNo.getAsLong();
            return l;
        }
    }

    TranslogWriter getCurrent() {
        return this.current;
    }

    List<TranslogReader> getReaders() {
        return this.readers;
    }

    public static String createEmptyTranslog(Path location, long initialGlobalCheckpoint, ShardId shardId, long primaryTerm) throws IOException {
        ChannelFactory channelFactory = FileChannel::open;
        return Translog.createEmptyTranslog(location, initialGlobalCheckpoint, shardId, channelFactory, primaryTerm);
    }

    static String createEmptyTranslog(Path location, long initialGlobalCheckpoint, ShardId shardId, ChannelFactory channelFactory, long primaryTerm) throws IOException {
        return Translog.createEmptyTranslog(location, shardId, initialGlobalCheckpoint, primaryTerm, null, channelFactory);
    }

    public static String createEmptyTranslog(Path location, ShardId shardId, long initialGlobalCheckpoint, long primaryTerm, @Nullable String translogUUID, @Nullable ChannelFactory factory) throws IOException {
        IOUtils.rm((Path[])new Path[]{location});
        Files.createDirectories(location, new FileAttribute[0]);
        long generation = 1L;
        long minTranslogGeneration = 1L;
        ChannelFactory channelFactory = factory != null ? factory : FileChannel::open;
        String uuid = org.elasticsearch.common.Strings.hasLength(translogUUID) ? translogUUID : UUIDs.randomBase64UUID();
        Path checkpointFile = location.resolve(CHECKPOINT_FILE_NAME);
        Path translogFile = location.resolve(Translog.getFilename(1L));
        Checkpoint checkpoint = Checkpoint.emptyTranslogCheckpoint(0L, 1L, initialGlobalCheckpoint, 1L);
        Checkpoint.write(channelFactory, checkpointFile, checkpoint, StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW);
        TranslogWriter writer = TranslogWriter.create(shardId, uuid, 1L, translogFile, channelFactory, TranslogConfig.EMPTY_TRANSLOG_BUFFER_SIZE, 1L, initialGlobalCheckpoint, () -> {
            throw new UnsupportedOperationException();
        }, () -> {
            throw new UnsupportedOperationException();
        }, primaryTerm, new TragicExceptionHolder(), seqNo -> {
            throw new UnsupportedOperationException();
        }, BigArrays.NON_RECYCLING_INSTANCE, DiskIoBufferPool.INSTANCE, (d, s, l) -> {});
        writer.close();
        return uuid;
    }

    /*
     * Uses 'sealed' constructs - enablewith --sealed true
     */
    public static abstract class Operation
    implements Writeable {
        protected final long seqNo;
        protected final long primaryTerm;

        protected Operation(long seqNo, long primaryTerm) {
            this.seqNo = seqNo;
            this.primaryTerm = primaryTerm;
        }

        public abstract Type opType();

        public abstract long estimateSize();

        public final long seqNo() {
            return this.seqNo;
        }

        public final long primaryTerm() {
            return this.primaryTerm;
        }

        public static Operation readOperation(StreamInput input) throws IOException {
            Type type = Type.fromId(input.readByte());
            return switch (type) {
                default -> throw new IncompatibleClassChangeError();
                case Type.CREATE, Type.INDEX -> Index.readFrom(input);
                case Type.DELETE -> Delete.readFrom(input);
                case Type.NO_OP -> new NoOp(input);
            };
        }

        @Override
        public final void writeTo(StreamOutput out) throws IOException {
            out.writeByte(this.opType().id());
            this.writeBody(out);
        }

        protected abstract void writeBody(StreamOutput var1) throws IOException;

        public static enum Type {
            CREATE(1),
            INDEX(2),
            DELETE(3),
            NO_OP(4);

            private final byte id;

            private Type(byte id) {
                this.id = id;
            }

            public byte id() {
                return this.id;
            }

            public static Type fromId(byte id) {
                return switch (id) {
                    case 1 -> CREATE;
                    case 2 -> INDEX;
                    case 3 -> DELETE;
                    case 4 -> NO_OP;
                    default -> throw new IllegalArgumentException("no type mapped for [" + id + "]");
                };
            }
        }
    }

    public static class Location
    implements Comparable<Location> {
        public static Location EMPTY = new Location(0L, 0L, 0);
        public final long generation;
        public final long translogLocation;
        public final int size;

        public Location(long generation, long translogLocation, int size) {
            this.generation = generation;
            this.translogLocation = translogLocation;
            this.size = size;
        }

        public String toString() {
            return "[generation: " + this.generation + ", location: " + this.translogLocation + ", size: " + this.size + "]";
        }

        @Override
        public int compareTo(Location o) {
            if (this.generation == o.generation) {
                return Long.compare(this.translogLocation, o.translogLocation);
            }
            return Long.compare(this.generation, o.generation);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Location location = (Location)o;
            if (this.generation != location.generation) {
                return false;
            }
            if (this.translogLocation != location.translogLocation) {
                return false;
            }
            return this.size == location.size;
        }

        public int hashCode() {
            int result = Long.hashCode(this.generation);
            result = 31 * result + Long.hashCode(this.translogLocation);
            result = 31 * result + this.size;
            return result;
        }
    }

    public static interface Snapshot
    extends Closeable {
        public static final Snapshot EMPTY = new Snapshot(){

            @Override
            public void close() {
            }

            @Override
            public int totalOperations() {
                return 0;
            }

            @Override
            public Operation next() {
                return null;
            }
        };

        public int totalOperations();

        default public int skippedOperations() {
            return 0;
        }

        public Operation next() throws IOException;
    }

    private static final class SeqNoFilterSnapshot
    implements Snapshot {
        private final Snapshot delegate;
        private int filteredOpsCount;
        private final long fromSeqNo;
        private final long toSeqNo;

        SeqNoFilterSnapshot(Snapshot delegate, long fromSeqNo, long toSeqNo) {
            assert (fromSeqNo <= toSeqNo) : "from_seq_no[" + fromSeqNo + "] > to_seq_no[" + toSeqNo + "]";
            this.delegate = delegate;
            this.fromSeqNo = fromSeqNo;
            this.toSeqNo = toSeqNo;
        }

        @Override
        public int totalOperations() {
            return this.delegate.totalOperations();
        }

        @Override
        public int skippedOperations() {
            return this.filteredOpsCount + this.delegate.skippedOperations();
        }

        @Override
        public Operation next() throws IOException {
            Operation op;
            while ((op = this.delegate.next()) != null) {
                if (this.fromSeqNo <= op.seqNo() && op.seqNo() <= this.toSeqNo) {
                    return op;
                }
                ++this.filteredOpsCount;
            }
            return null;
        }

        @Override
        public void close() throws IOException {
            this.delegate.close();
        }
    }

    public static final class TranslogGeneration {
        public final String translogUUID;
        public final long translogFileGeneration;

        public TranslogGeneration(String translogUUID, long translogFileGeneration) {
            this.translogUUID = translogUUID;
            this.translogFileGeneration = translogFileGeneration;
        }
    }

    public static enum Durability {
        ASYNC,
        REQUEST;

    }

    public static final class NoOp
    extends Operation {
        private final String reason;

        public String reason() {
            return this.reason;
        }

        private NoOp(StreamInput in) throws IOException {
            this(in.readLong(), in.readLong(), in.readString());
        }

        public NoOp(long seqNo, long primaryTerm, String reason) {
            super(seqNo, primaryTerm);
            assert (seqNo > -1L);
            assert (primaryTerm >= 0L);
            assert (reason != null);
            this.reason = reason;
        }

        @Override
        public void writeBody(StreamOutput out) throws IOException {
            out.writeLong(this.seqNo);
            out.writeLong(this.primaryTerm);
            out.writeString(this.reason);
        }

        @Override
        public Operation.Type opType() {
            return Operation.Type.NO_OP;
        }

        @Override
        public long estimateSize() {
            return 2 * this.reason.length() + 16;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            NoOp that = (NoOp)obj;
            return this.seqNo == that.seqNo && this.primaryTerm == that.primaryTerm && this.reason.equals(that.reason);
        }

        public int hashCode() {
            return 961 * Long.hashCode(this.seqNo) + 31 * Long.hashCode(this.primaryTerm) + this.reason().hashCode();
        }

        public String toString() {
            return "NoOp{seqNo=" + this.seqNo + ", primaryTerm=" + this.primaryTerm + ", reason='" + this.reason + "'}";
        }
    }

    public static final class Delete
    extends Operation {
        private static final int FORMAT_6_0 = 4;
        public static final int FORMAT_NO_PARENT = 5;
        public static final int FORMAT_NO_VERSION_TYPE = 6;
        public static final int FORMAT_NO_DOC_TYPE = 7;
        public static final int SERIALIZATION_FORMAT = 7;
        private final String id;
        private final long version;

        private static Delete readFrom(StreamInput in) throws IOException {
            int format = in.readVInt();
            assert (format >= 4) : "format was: " + format;
            if (format < 7) {
                in.readString();
            }
            String id = in.readString();
            if (format < 7) {
                String docType = in.readString();
                assert (docType.equals("_id")) : docType + " != _id";
                in.readBytesRef();
            }
            long version = in.readLong();
            if (format < 6) {
                in.readByte();
            }
            long seqNo = in.readLong();
            long primaryTerm = in.readLong();
            return new Delete(id, seqNo, primaryTerm, version);
        }

        public Delete(Engine.Delete delete, Engine.DeleteResult deleteResult) {
            this(delete.id(), deleteResult.getSeqNo(), delete.primaryTerm(), deleteResult.getVersion());
        }

        public Delete(String id, long seqNo, long primaryTerm) {
            this(id, seqNo, primaryTerm, -3L);
        }

        public Delete(String id, long seqNo, long primaryTerm, long version) {
            super(seqNo, primaryTerm);
            this.id = Objects.requireNonNull(id);
            this.version = version;
        }

        @Override
        public Operation.Type opType() {
            return Operation.Type.DELETE;
        }

        @Override
        public long estimateSize() {
            return 2 * this.id.length() + 24;
        }

        public String id() {
            return this.id;
        }

        public long version() {
            return this.version;
        }

        @Override
        public void writeBody(StreamOutput out) throws IOException {
            int format = out.getTransportVersion().onOrAfter(TransportVersion.V_8_0_0) ? 7 : 6;
            out.writeVInt(format);
            if (format < 7) {
                out.writeString("_doc");
            }
            out.writeString(this.id);
            if (format < 7) {
                out.writeString("_id");
                out.writeBytesRef(Uid.encodeId(this.id));
            }
            out.writeLong(this.version);
            out.writeLong(this.seqNo);
            out.writeLong(this.primaryTerm);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Delete delete = (Delete)o;
            return this.id.equals(delete.id) && this.seqNo == delete.seqNo && this.primaryTerm == delete.primaryTerm && this.version == delete.version;
        }

        public int hashCode() {
            int result = this.id.hashCode();
            result += 31 * Long.hashCode(this.seqNo);
            result = 31 * result + Long.hashCode(this.primaryTerm);
            result = 31 * result + Long.hashCode(this.version);
            return result;
        }

        public String toString() {
            return "Delete{id='" + this.id + "', seqNo=" + this.seqNo + ", primaryTerm=" + this.primaryTerm + ", version=" + this.version + "}";
        }
    }

    public static final class Index
    extends Operation {
        public static final int FORMAT_NO_PARENT = 9;
        public static final int FORMAT_NO_VERSION_TYPE = 10;
        public static final int FORMAT_NO_DOC_TYPE = 11;
        public static final int SERIALIZATION_FORMAT = 11;
        private final String id;
        private final long autoGeneratedIdTimestamp;
        private final long version;
        private final BytesReference source;
        private final String routing;

        private static Index readFrom(StreamInput in) throws IOException {
            int format = in.readVInt();
            assert (format >= 9) : "format was: " + format;
            String id = in.readString();
            if (format < 11) {
                in.readString();
            }
            BytesReference source = in.readBytesReference();
            String routing = in.readOptionalString();
            long version = in.readLong();
            if (format < 10) {
                in.readByte();
            }
            long autoGeneratedIdTimestamp = in.readLong();
            long seqNo = in.readLong();
            long primaryTerm = in.readLong();
            return new Index(id, seqNo, primaryTerm, version, source, routing, autoGeneratedIdTimestamp);
        }

        public Index(Engine.Index index, Engine.IndexResult indexResult) {
            this(index.id(), indexResult.getSeqNo(), index.primaryTerm(), indexResult.getVersion(), index.source(), index.routing(), index.getAutoGeneratedIdTimestamp());
        }

        public Index(String id, long seqNo, long primaryTerm, long version, BytesReference source, String routing, long autoGeneratedIdTimestamp) {
            super(seqNo, primaryTerm);
            this.id = id;
            this.source = source;
            this.version = version;
            this.routing = routing;
            this.autoGeneratedIdTimestamp = autoGeneratedIdTimestamp;
        }

        @Override
        public Operation.Type opType() {
            return Operation.Type.INDEX;
        }

        @Override
        public long estimateSize() {
            return 2 * this.id.length() + this.source.length() + (this.routing != null ? 2 * this.routing.length() : 0) + 32;
        }

        public String id() {
            return this.id;
        }

        public String routing() {
            return this.routing;
        }

        public BytesReference source() {
            return this.source;
        }

        public long version() {
            return this.version;
        }

        @Override
        public void writeBody(StreamOutput out) throws IOException {
            int format = out.getTransportVersion().onOrAfter(TransportVersion.V_8_0_0) ? 11 : 10;
            out.writeVInt(format);
            out.writeString(this.id);
            if (format < 11) {
                out.writeString("_doc");
            }
            out.writeBytesReference(this.source);
            out.writeOptionalString(this.routing);
            out.writeLong(this.version);
            out.writeLong(this.autoGeneratedIdTimestamp);
            out.writeLong(this.seqNo);
            out.writeLong(this.primaryTerm);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Index index = (Index)o;
            if (this.version != index.version || this.seqNo != index.seqNo || this.primaryTerm != index.primaryTerm || !this.id.equals(index.id) || this.autoGeneratedIdTimestamp != index.autoGeneratedIdTimestamp || !this.source.equals(index.source)) {
                return false;
            }
            return Objects.equals(this.routing, index.routing);
        }

        public int hashCode() {
            int result = this.id.hashCode();
            result = 31 * result + Long.hashCode(this.seqNo);
            result = 31 * result + Long.hashCode(this.primaryTerm);
            result = 31 * result + Long.hashCode(this.version);
            result = 31 * result + this.source.hashCode();
            result = 31 * result + (this.routing != null ? this.routing.hashCode() : 0);
            result = 31 * result + Long.hashCode(this.autoGeneratedIdTimestamp);
            return result;
        }

        public String toString() {
            return "Index{id='" + this.id + "', seqNo=" + this.seqNo + ", primaryTerm=" + this.primaryTerm + ", version=" + this.version + ", autoGeneratedIdTimestamp=" + this.autoGeneratedIdTimestamp + "}";
        }

        public long getAutoGeneratedIdTimestamp() {
            return this.autoGeneratedIdTimestamp;
        }
    }
}

